# /// script
# requires-python = ">=3.13"
# dependencies = [
#     "kaleido==1.2.0",
#     "marimo",
#     "numpy==2.4.2",
#     "pandas==3.0.1",
#     "plotly==6.5.2",
#     "scipy==1.17.0",
#     "statsmodels==0.14.6",
# ]
# ///

import marimo

__generated_with = "0.20.1"
app = marimo.App(width="medium")


@app.cell(hide_code=True)
def _(mo):
    mo.hstack(
        [
            mo.image(
                src="http://modernmacro.org/resources/Vivaldo/figures/OldTests/RIR.png",
                width=200,
            ),
            mo.image(
                src="http://modernmacro.org/resources/Vivaldo/figures/iscte.png",
                width=300,
            ),
        ],
        align="center",
    )
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    <style>
    .marimo-cell {
        border: none !important;
        box-shadow: none !important;
        margin: 0 !important;
        padding: 0 !important;
    }

    .marimo-cell-content {
        padding: 0.5rem 0 !important;
    }
    .admonition {
    margin-top: -0.99rem !important;
    margin-bottom: -0.99rem !important;
    }
    h1 {
        margin-bottom: -1.2rem !important;
    }
    h2 {
        margin-bottom: -1.2rem !important;
    }
    /* Reduce spacing before bullet lists */
    ul {
        margin-top: -0.3rem !important;
    }
    /* Optionally reduce spacing between list items */
    li {
        margin-bottom: 0.2rem !important;
    }
    /* Increase font size for all text in the document */
    body {
        font-size: 16px;
    }

    /* Increase font size for all headers */
    h1 {
        font-size: 36px !important;
    }

    h2 {
        font-size: 28px !important;
    }

    h3 {
        font-size: 25px !important;
    }
    </style>
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    # The HP filter & Impulse Response Functions
    <br>
    **February 23, 2026**

    **Vivaldo Mendes**
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ### Packages used in this notebook
    """)
    return


@app.cell
def _():
    import marimo as mo
    import numpy as np
    import pandas as pd
    from scipy.optimize import fsolve 
    from scipy.sparse.linalg import spsolve
    from scipy.sparse import diags
    import plotly.graph_objects as go                      
    from plotly.subplots import make_subplots
    import plotly.express as px
    import plotly.io as pio 
    pio.templates.default = "plotly"
    import kaleido
    pio.defaults.default_format = "svg"
    from datetime import date, datetime
    from numpy.linalg import eigvals, inv
    from statsmodels.tsa.stattools import acf, ccf as statsmodels_ccf
    import warnings
    warnings.filterwarnings("ignore")
    import os
    os.chdir(os.path.dirname(__file__))
    return diags, go, mo, np, pd, spsolve


@app.cell
def _():
    # import os
    #os.getcwd()
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ## 1. Functions
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    /// attention

    Students are not expected to understand all the small details of the elements that comprise the functions in this section, but they do need to know how to use them in the sections below. Students do not need to study these functions; however, they need to understand very well what they do (what they are used for).
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    The functions below will be used in this notebook to compute the Hodrick-Prescott trend (HP trend). The only indispensable function is <kbd>hp_trend</kbd>. But to make our lives easier, we also wrote <kbd>hp_trend_df</kbd>, which allows us to apply the former to an entire dataframe. The final function <kbd>df_arith_operations</kbd> is used to implement simple arithmetic operations over two dataframes (like <kbd>+</kbd> , <kbd>-</kbd> , <kbd>*</kbd> , <kbd>\</kbd> , etc.), by skipping the column with the dates.

    A complete understanding of the code in these three functions will not be required in this course. But students should understand how (and why) we will use them in this notebook.
    """)
    return


@app.cell
def _(diags, np, spsolve):
    def hp_trend(mydata: np.ndarray, λ: float) -> np.ndarray:
        """
        Computes the Hodrick-Prescott trend trend.

        Builds the symmetric band matrix A from the first-order conditions
        of the HP minimization problem using sparse diagonals:

            A = diags([diag2, diag1, diag0, diag1, diag2], [-2,-1,0,1,2])

        Parameters
        ----------
        mydata : 1-D numpy array of float64
        λ      : smoothing parameter (e.g. 1600 for quarterly data)

        Returns
        -------
        trend : 1-D numpy array (the HP trend τ)
        """
        n = len(mydata)
        assert n >= 4, "Need at least 4 observations."

        diag2 = λ * np.ones(n - 2)
        diag1 = np.concatenate([[-2 * λ], -4 * λ * np.ones(n - 3), [-2 * λ]])
        diag0 = np.concatenate(
            [[1 + λ], [1 + 5 * λ], (1 + 6 * λ) * np.ones(n - 4), [1 + 5 * λ], [1 + λ]]
        )

        A = diags([diag2, diag1, diag0, diag1, diag2], [-2, -1, 0, 1, 2], format="csc")

        return spsolve(A, mydata)

    return (hp_trend,)


@app.cell
def _(hp_trend, pd):
    def hp_trend_df(df: pd.DataFrame, λ: float) -> pd.DataFrame:
        """
        Applies hp_trend to every numeric column of a DataFrame,
        preserving the first (date) column unchanged.

        Parameters
        ----------
        df : DataFrame whose first column is a date/period object
        λ  : HP smoothing parameter

        Returns
        -------
        trended_df : DataFrame with trend values (same column names)
        """
        colnames = df.columns.tolist()
        trended_df = pd.DataFrame()
        trended_df[colnames[0]] = df[colnames[0]]

        for col in colnames[1:]:
            trended_df[col] = hp_trend(df[col].values.astype(float), λ)

        return trended_df

    return (hp_trend_df,)


@app.cell
def _(pd):
    def df_arith_operations(df1: pd.DataFrame, df2: pd.DataFrame, op) -> pd.DataFrame:
        """
        Applies a binary operation column-wise between two DataFrames,
        skipping the first (date) column.

        Example — cycle = original − trend:
            USdata_cycles = df_arith_operations(USdata, USdata_trends, np.subtract)

        Parameters
        ----------
        df1, df2 : DataFrames with identical column names
        op       : a numpy ufunc or any callable f(a, b) → array,
                   e.g. np.subtract, np.add, np.divide, np.multiply

        Returns
        -------
        result_df : DataFrame with the operation applied column-wise
        """
        colnames = df1.columns.tolist()
        result_df = pd.DataFrame()
        result_df[colnames[0]] = df1[colnames[0]]

        for col in colnames[1:]:
            result_df[col] = op(df1[col].values, df2[col].values)

        return result_df

    return (df_arith_operations,)


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ## 2. Data
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ### Read the data
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    To read or load the data into the notebook, we use <kbd>pd.read_csv</kbd>. The CSV file <kbd>Data_US.csv</kbd> must be located in the same folder as this notebook:

    ```python
            USdata = pd.read_csv("Data_US.csv", parse_dates=["Quarters"])
    ```
    Usually, the <kbd>pandas</kbd> library is capable of inferring the correct date object (like quarters in the current case) and, therefore, to load the data set it will be enough to type the following code:

    ```python
            USdata = pd.read_csv("Data_US.csv")
    ```
    """)
    return


@app.cell
def _(pd, preview):
    USdata = pd.read_csv("Data_US.csv")  # , parse_dates=["Quarters"])
    preview(USdata)
    return (USdata,)


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ### Create sub-samples of your data
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    To work with a particular sub-period, stipulate an initial and an end date:

    ```python
    my_new_data = USdata[ (USdata["Quarters"] >= "1960-01-01") & (USdata["Quarters"] <= "1961-10-01") ]
    ```
    """)
    return


@app.cell
def _(USdata):
    my_new_data = USdata[ (USdata["Quarters"] >= "1960-01-01") & (USdata["Quarters"] <= "1961-10-01") ]
    my_new_data
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ### Doing some plotting
    """)
    return


@app.cell(hide_code=True)
def _(USdata, go):
    _fig = go.Figure()
    _fig.add_trace(
        go.Scatter(
            x=USdata["Quarters"], y=USdata["CPI"], mode="lines", name="Inflation (π)"
        )
    )
    _fig.add_trace(
        go.Scatter(
            x=USdata["Quarters"], y=USdata["UR"], mode="lines", name="Unemployment (u)"
        )
    )
    _fig.update_layout(
        height=460,
        title_text="Unemployment and inflation in the US: 1960Q1–2023Q4",
        title_x=0.5,
        hovermode="x",
        font_family="Calibri",
        font_size=14,
        legend=dict(x=0.03, y=0.95, bgcolor="rgba(0,0,0,0)"),
        xaxis_range=["1959-04-01", "2024-10-01"],
        margin=dict(l=70, r=60, t=70, b=60),
    )
    _fig
    return


@app.cell(hide_code=True)
def _(USdata, go):
    _fig2 = go.Figure()

    _fig2.add_trace(
        go.Scatter(
            x=USdata["Quarters"],
            y=USdata["CPI"],
            mode="lines",
            name="Inflation (π)",
            line=dict(color="blueviolet", width=2),
            yaxis="y2",
        )
    )
    _fig2.add_trace(
        go.Scatter(
            x=USdata["Quarters"],
            y=USdata["UR"],
            mode="lines",
            name="Unemployment (u)",
            line=dict(color="darkblue", width=2),
            yaxis="y2",
        )
    )
    _fig2.add_trace(
        go.Scatter(
            x=USdata["Quarters"],
            y=USdata["GDP"],
            mode="lines",
            name="GDP (y)",
            line=dict(color="maroon", width=2),
        )
    )

    _fig2.update_layout(
        margin=dict(l=70, r=60, t=70, b=60),
        height=460,
        title_text="GDP, Unemployment and Inflation in the US: 1960Q1–2023Q4",
        title_x=0.5,
        hovermode="x",
        font_family="Calibri",
        font_size=14,
        legend=dict(x=0.02, y=0.94, bgcolor="rgba(0,0,0,0)"),
        xaxis=dict(
            title="Quarterly observations",
            showgrid=True,
            range=["1959-04-01", "2024-10-01"],
        ),
        yaxis=dict(
            title=dict(text="Real GDP", font=dict(color="maroon")),
            tickfont=dict(color="maroon"),
            showgrid=False,
            zeroline=True,
        ),
        yaxis2=dict(
            title=dict(text="Unemployment, Inflation", font=dict(color="blue")),
            tickfont=dict(color="blue"),
            overlaying="y",
            side="right",
            showgrid=True,
            zeroline=False,
            range=[-2, 15],
        ),
    )
    _fig2
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### Inserting new columns & deleting existing columns
    In pandas we use `insert` to add columns at a specific position, and `drop` to remove columns.

    - **Inserting a new column** (e.g. log of GDP at position 2):
      ```python
            USdata.insert(2, "lnGDP", np.log(USdata["GDP"]))
      ```
    - **Deleting an existing column**:
      ```python
            USdata = USdata.drop(columns=["FFR"])
      ```
    """)
    return


@app.cell
def _(USdata, np):
    USdata.insert(2, "lnGDP", np.log(USdata["GDP"]))
    return


@app.cell
def _(USdata, preview):
    USdata
    preview(USdata)
    return


@app.cell
def _(USdata):
    USdata.drop(columns=["FFR"], inplace=True)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ### Creating an entirely new dataframe
    """)
    return


@app.cell
def _(USdata):
    USdata2 = USdata[["CPI", "UR", "GDP", "lnGDP"]].copy()
    USdata2
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **Exercise 1**

    Insert a new column into the dataframe by computing the logarithm of M2, suppress the display of its output and check that the operation was correctly implemented.
    """)
    return


@app.cell
def _():
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ## 3. The Hodrick-Prescott filter
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    This filter intends to separate a time-series $y_t$ into a trend $\tau_t$ and a cyclical component $\varphi_t$ such that:

    $$y_t = \tau_t + \varphi_t \tag{1}$$

    - The trend is the long-run component of the time series
    - The cyclical component is the short-run component of the time series
    - Therefore, from (1) we get:

    $$\varphi_t = y_t - \tau_t \tag{2}$$

    The trick is to minimize eq. (2) subject to some given constraint on $\varphi_t$. The HP trend is given by the following minimization problem with respect to $\tau_t$:

    $$\min _{\tau_{t}} \left\{ {\cal{L(\tau)}} =\sum_{t=1}^{n} \underbrace{\left(y_{t}-\tau_{t}\right)^{2}}_{=\varphi_t^2} +\ {\color{red}\lambda} \sum_{t=2}^{n-1} \underbrace{\left[\left(\tau_{t+1}-\tau_{t}\right)-\left(\tau_{t}-\tau_{t-1}\right)\right]^{2}}_{\text{constraint}}\right\} \tag{3}$$

    where $\lambda$ is the parameter that we set to obtain the <span style='color:red'>desired</span> smoothness in the trend.
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ### Setting the value of $\lambda$
    """)
    return


@app.cell
def _(λ_slider):
    λ = λ_slider.value
    return (λ,)


@app.cell(hide_code=True)
def _(mo):
    λ_slider = mo.ui.slider(
        start=0,
        stop=150000,
        step=800,
        value=1600,
        label="λ",
        show_value=True,
        full_width=True,
    )
    λ_slider.style({"width": "500px"})
    return (λ_slider,)


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### HP trend on a single variable/column name

    To compute the HP trend on a single column (`lnGDP`):

    ```python
            GDP_trend  = hp_trend(USdata["lnGDP"].values, λ)
            GDP_cycles = USdata["lnGDP"].values - GDP_trend
    ```
    """)
    return


@app.cell
def _(USdata, hp_trend, λ):
    GDP_trend = hp_trend(USdata.lnGDP.values, λ)
    GDP_cycles = USdata.lnGDP.values - GDP_trend
    return GDP_cycles, GDP_trend


@app.cell(hide_code=True)
def _(GDP_trend, USdata, go):
    _fig3 = go.Figure()
    _fig3.add_trace(
        go.Scatter(x=USdata.Quarters, y=USdata.lnGDP, mode="lines", name="GDP")
    )
    _fig3.add_trace(
        go.Scatter(x=USdata.Quarters, y=GDP_trend, mode="lines", name="HP_trend")
    )
    _fig3.update_layout(
        height=450,
        hovermode="x",
        title_text="GDP versus GDP trend (US: 1960Q1–2023Q4)",
        title_x=0.5,
        xaxis=dict(title="Quarters", range=["1959-04-01", "2024-10-01"]),
        yaxis_title="Billion dollars (logarithmic)",
        margin=dict(l=70, r=60, t=70, b=60),
    )
    _fig3
    return


@app.cell(hide_code=True)
def _(GDP_cycles, USdata, go):
    _fig4 = go.Figure()
    _fig4.add_trace(
        go.Bar(x=USdata.Quarters, y=GDP_cycles, name="HP_GDP", marker_color="Navy")
    )
    _fig4.update_layout(
        height=450,
        hovermode="x",
        bargap=0.4,
        title_text="The Output-gap: HP trended business cycles (US: 1960Q1–2023Q4)",
        title_x=0.5,
        xaxis=dict(title="Quarters", range=["1959-04-01", "2024-10-01"]),
        yaxis_title="Percentage deviations",
        margin=dict(l=70, r=60, t=70, b=60),
    )
    _fig4
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ### The HP trend on an entire data set

    There are two significant shortcomings in the steps we undertook above:

    - Firstly, it is terribly inefficient: if we have a dataframe with, e.g., 20 variables, we must repeat the same procedure 20 times.

    - Secondly, as the HP-trend can not trend non-numerical data (like dates in the first column of the original data frame: "Quarters"), this creates further problems if we want to work with dataframes (which we should).

    By using the <kbd>hp_trend_df</kbd> function instead of <kbd>hp_trend</kbd>, the two problems above will vanish. The function <kbd>hp_trend_df</kbd> uses the main function <kbd>hp_trend</kbd> and has three useful properties:

    - It allows the trend to be applied to an entire dataframe and produces a new dataframe as the output.

    - It keeps the first non-numeric column (date object: year, quarter, month, etc.) of the original data frame as the first column of the new data frame.

    - This is extremely convenient because we will have all the functionalities of a data frame when producing further operations on the new trended results.

    To compute the HP trend associated with each variable (column) of the entire data set, all we need to do is to apply the following code:

    ```python
        USdata_trends = hp_trend_df(USdata, λ)
    ```
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ### Computing the trend
    """)
    return


@app.cell
def _(USdata, hp_trend_df, preview, λ):
    USdata_trends = hp_trend_df(USdata, λ)
    preview(USdata_trends)
    return (USdata_trends,)


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ### Computing the cycles

    According to eq. (2), the cycles are computed as the difference between the original time series $(y_t)$ and its long term trend $(\tau_t)$:

    $$\varphi_t = y_t - \tau_t \tag{2a}$$

    As $y_t$ and $\tau_t$ are two dataframes (respectively,<kbd>USdata</kbd> and <kbd>USdata_trends</kbd>), the computation of the difference between the two dataframes (column by column) can be easily accomplished by using some simple binary operations involving the two data frames. The function <kbd>df_arith_operations</kbd>) can do that very easily, just by typing:

    ```python
        USdata_cycles = df_arith_operations(USdata, USdata_trends, np.subtract)
    ```

    This function will pass the binary operation <kbd>np.subtract</kbd> as a function and it will apply it column-wise accross the two data frames, **skipping the first column (the date object)**.
    """)
    return


@app.cell
def _(USdata, USdata_trends, df_arith_operations, np, preview):
    USdata_cycles = df_arith_operations(USdata, USdata_trends, np.subtract)
    preview(USdata_cycles)
    return (USdata_cycles,)


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ### Some plotting
    """)
    return


@app.cell(hide_code=True)
def _(USdata, USdata_cycles, go):
    _fig5 = go.Figure()
    _fig5.add_trace(
        go.Bar(x=USdata.Quarters, y=USdata_cycles.lnGDP, marker_color="Navy")
    )
    _fig5.update_layout(
        height=450,
        hovermode="x",
        bargap=0.4,
        title_text="The Output-gap using the HP trend (US: 1960Q1–2023Q4)",
        title_x=0.5,
        xaxis=dict(title="Quarters", range=["1959-04-01", "2024-10-01"]),
        yaxis_title="Percentage deviations",
        margin=dict(l=70, r=60, t=70, b=60),
    )
    _fig5
    return


@app.cell(hide_code=True)
def _(USdata, USdata_cycles, go):
    _fig6 = go.Figure()
    _fig6.add_trace(
        go.Bar(x=USdata.Quarters, y=USdata_cycles.GDP, marker_color="maroon")
    )
    _fig6.update_layout(
        height=450,
        hovermode="x",
        bargap=0.4,
        title_text="Incorrect measure of the Output-gap in the US: 1960Q1–2023Q4",
        title_x=0.5,
        xaxis=dict(title="Quarters", range=["1959-04-01", "2024-10-01"]),
        yaxis_title="Percentage deviations",
        margin=dict(l=70, r=60, t=70, b=60),
    )
    _fig6
    return


@app.cell(hide_code=True)
def _(USdata, USdata_cycles, go):
    _fig7 = go.Figure()
    _fig7.add_trace(
        go.Scatter(
            x=USdata.Quarters,
            y=USdata_cycles.CPI,
            mode="markers+lines",
            marker=dict(symbol="circle", size=5.5, color="darkblue"),
            line=dict(width=0.3),
            name="CPI",
        )
    )
    _fig7.update_layout(
        height=450,
        title_text="The inflation-gap in the US: 1960Q1–2023Q4",
        title_x=0.5,
        hovermode="x",
        margin=dict(l=70, r=60, t=70, b=60),
        xaxis=dict(
            title="Quarterly observations",
            tickformat="%Y",
            range=["1959-04-01", "2024-10-01"],
        ),
        yaxis=dict(title="Percentage deviations", range=[-6, 7]),
    )
    _fig7
    return


@app.cell(hide_code=True)
def _(USdata, USdata_trends, go):
    _fig8 = go.Figure()
    _fig8.add_trace(
        go.Scatter(
            x=USdata.Quarters,
            y=USdata_trends.UR,
            mode="lines+markers",
            marker=dict(symbol="circle", size=4.5, color="darkred"),
            line=dict(width=0.3),
            name="NUR (uⁿ)",
        )
    )
    _fig8.add_trace(
        go.Scatter(
            x=USdata["Quarters"],
            y=USdata["UR"],
            mode="lines+markers",
            marker=dict(symbol="circle", size=5, color="blue"),
            line=dict(width=0.3),
            name="Unemployment rate (u)",
        )
    )
    _fig8.update_layout(
        margin=dict(l=70, r=60, t=70, b=60),
        height=450,
        font_family="Calibri",
        font_size=15,
        title_text=(
            "The Unemployment Rate vs the Natural Unemployment Rate "
            "(trend) in the US: 1960Q1–2023Q4"
        ),
        title_x=0.5,
        hovermode="x",
        legend=dict(x=0.02, y=0.94),
        xaxis=dict(
            title="Quarterly observations",
            tickformat="%Y",
            range=["1959-04-01", "2024-10-01"],
        ),
        yaxis_title="Percentage points",
    )
    _fig8
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **Exercise 2**

    Produce a plot involving the **logarithmic** value of the Unemployment Rate and its long-term trend. Copy the code from the previous plot and introduce the required changes: variable names, title, and variable labels.
    """)
    return


@app.cell
def _():
    # Exercise 2 — your code here
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    👉 **Exercise 3**

    In the slides we saw that if a macroeconomic variable has a clear trend, the HP trend should be applied to the **logarithmic** value of the original observations. Using the case of real GDP, confirm that dividing its HP cycles (obtained *without* logarithms) by its corresponding trend (also without logarithms) yields the same "correct" measure    of business cycles as applying logarithms to the original observations (numerically or graphically).
    """)
    return


@app.cell
def _():
    # Exercise 3 — your code here
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ## 4. Impulse Response Functions
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ### AR(1)
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    Consider the following AR(1) process:

    $$y_{t+1} = \color{blueviolet}{a} \cdot y_t + \varepsilon_{t+1} \tag{1}$$

    Assume $y_1 = 0$, $\varepsilon_2 = 1$, and $\varepsilon_t = 0$ for all $t \neq 2$.

    This implies $y_2 = 1$. What happens next, if there are no more shocks?
    The IRF of $y$ provides the answer. The dynamics depend crucially on $\color{blueviolet}{a}$:

    $$\color{blueviolet}{a} = \{0,\, 0.5,\, 0.9,\, 0.99,\, 1,\, 1.01\}$$
    """)
    return


@app.cell(hide_code=True)
def _(go, np):
    # AR(1) — single case a = 0.5
    _n = 20
    _y1 = 0.0
    _ε = np.zeros(_n)
    _ε[1] = 1.0
    _a = 0.5
    _y = np.zeros(_n)
    _y[0] = _y1
    for _t in range(_n - 1):
        _y[_t + 1] = _a * _y[_t] + _ε[_t + 1]

    _fig9 = go.Figure()
    _fig9.add_trace(
        go.Scatter(
            y=_y,
            mode="markers+lines",
            marker=dict(symbol="circle", size=6, color="blue"),
            line=dict(width=0.3),
        )
    )
    _fig9.update_layout(
        margin=dict(l=70, r=60, t=70, b=60),
        title_text="A stochastic AR(1) process",
        title_x=0.5,
        hovermode="x",
        height=450,
        # Exercise 3 — your code here
    )
    _fig9
    return


@app.cell(hide_code=True)
def _(go, np):
    # AR(1) — six values of a
    _𝕟 = 30
    _𝕒 = (0.0, 0.5, 0.9, 0.99, 1.0, 1.01)
    𝕖 = np.zeros((6, _𝕟))
    𝕖[:, 1] = 1.0
    _𝕪 = np.zeros((6, _𝕟))

    for 𝕜 in range(6):
        for 𝕥 in range(_𝕟 - 1):
            _𝕪[𝕜, 𝕥 + 1] = _𝕒[𝕜] * _𝕪[𝕜, 𝕥] + 𝕖[𝕜, 𝕥 + 1]

    _colors = ["blue", "red", "grey", "purple", "darkblue", "orange"]
    _names = ["𝒂 = 0", "𝒂 = 0.5", "𝒂 = 0.9", "𝒂 = 0.99", "𝒂 = 1.0", "𝒂 = 1.01"]

    _fig10 = go.Figure()
    for _k in range(6):
        _fig10.add_trace(
            go.Scatter(
                y=_𝕪[_k],
                mode="markers+lines",
                marker=dict(size=6),
                line=dict(width=0.5, color=_colors[_k]),
                name=_names[_k],
            )
        )
    _fig10.update_layout(
        margin=dict(l=70, r=60, t=70, b=60),
        title_text="Impulse Response Functions (IRF) from an AR(1) process",
        title_x=0.5,
        hovermode="x",
        height=500,
    )
    _fig10
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ### AR(2)
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    Consider the AR(2) process:

    $$y_{t+1} = \color{red}{a}\, y_t + \color{blueviolet}{b}\, y_{t-1} + \varepsilon_{t+1},
    \quad \varepsilon_t \sim N(0,1) \tag{2}$$

    Assume $y_1 = y_2 = 0$, $\varepsilon_3 = 1$, $\varepsilon_t = 0$ for all $t \neq 3$,
    so $y_3 = 1$.

    Fix $\color{red}{b = -0.9}$ and let $\color{blueviolet}{a} \in \{1.85,\, 1.895,\, 1.9,\, 1.9005\}$.
    """)
    return


@app.cell(hide_code=True)
def _(go, np):
    # AR(2) — four values of a, b = -0.9
    _𝐍 = 99
    _𝐚 = (1.85, 1.895, 1.9, 1.9005)
    𝐛 = -0.9
    _𝛜 = np.zeros((4, _𝐍 + 1))
    _𝛜[:, 2] = 1.0
    _𝐲 = np.zeros((4, _𝐍 + 1))

    for _k in range(4):
        for _t in range(_𝐍 - 1):
            _𝐲[_k, _t + 2] = _𝐚[_k] * _𝐲[_k, _t + 1] + 𝐛 * _𝐲[_k, _t] + _𝛜[_k, _t + 2]

    _colors2 = ["blue", "navy", "gray", "darkorange"]
    _names2 = ["𝒂 = 1.85", "𝒂 = 1.895", "𝒂 = 1.9", "𝒂 = 1.9005"]

    _fig11 = go.Figure()
    for _k in range(4):
        _fig11.add_trace(
            go.Scatter(
                y=_𝐲[_k],
                mode="markers+lines",
                marker=dict(size=5),
                line=dict(width=0.5, color=_colors2[_k]),
                name=_names2[_k],
            )
        )
    _fig11.update_layout(
        margin=dict(l=70, r=60, t=70, b=60),
        title_text="Impulse Response Functions (IRF) from an AR(2) process",
        title_x=0.5,
        hovermode="x",
        height=500,
        xaxis_range=[-1, 102],
    )
    _fig11
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ### VAR(3)
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    A similar reasoning applies to the general model:

    $$X_{t+1} = A + B X_t + C \varepsilon_{t+1} \tag{3}$$

    where $B, C$ are $n \times n$ matrices and $X_{t+1}, X_t, A, \varepsilon_{t+1}$ are $n \times 1$ vectors.

    Consider the following VAR(3):

    $$X_{t+1} = \begin{bmatrix} z_{t+1} \\ w_{t+1} \\ v_{t+1} \end{bmatrix} \tag{4}$$

    with matrices:

    $$A = \begin{bmatrix} 0 \\ 0 \\ 0 \end{bmatrix}, \quad
    B = \begin{bmatrix} 0.97 & 0.10 & -0.05 \\ -0.3 & 0.8 & 0.05 \\ 0.01 & -0.04 & 0.96 \end{bmatrix}, \quad
    C = \begin{bmatrix} \color{blueviolet}{1.0} & 0 & 0 \\ 0 & 0 & 0 \\ 0 & 0 & 0 \end{bmatrix} \tag{5}$$

    Initial conditions: $X_1 = [0, 0, 0]$. The shock hits only $z_t$ (blue entry in $C$) at $t = 3$.
    """)
    return


@app.cell
def _(np):
    𝐴 = np.zeros((3, 1))
    𝐵 = np.array([[0.97, 0.10, -0.05], [-0.30, 0.80, 0.05], [0.01, -0.04, 0.96]])
    𝐶 = np.array([[1.0, 0.0, 0.0], [0.0, 0.0, 0.0], [0.0, 0.0, 0.0]])
    return A, B, C


@app.cell
def _(mo):
    𝑧1_slider = mo.ui.slider(
        start=0.0,
        stop=5.0,
        step=1.0,
        value=1.0,
        label="𝑧₁",
        show_value=True,
    )
    𝑧1_slider
    return (z1_slider,)


@app.cell(hide_code=True)
def _(A, B, C, go, np, z1_slider):
    𝑁 = 40
    𝑋1 = np.zeros((3, 1))
    _𝜀 = np.zeros((3, 𝑁))
    _𝜀[0, 2] = float(𝑧1_slider.value)
    𝑋 = np.zeros((3, 𝑁))
    𝑋[:, 0:1] = 𝑋1

    for _t in range(𝑁 - 1):
        𝑋[:, _t + 1 : _t + 2] = 𝐴 + 𝐵 @ 𝑋[:, _t : _t + 1] + 𝐶 @ _𝜀[:, _t + 1 : _t + 2]

    _names3 = ["𝒛(𝒕)", "𝒘(𝒕)", "𝒗(𝒕)"]
    _colors3 = ["blue", "gray", "navy"]

    _fig12 = go.Figure()
    for _i in range(3):
        _fig12.add_trace(
            go.Scatter(
                y=𝑋[_i],
                mode="markers+lines",
                marker=dict(symbol="circle", size=6),
                line=dict(width=0.3, color=_colors3[_i]),
                name=_names3[_i],
            )
        )
    _fig12.update_layout(
        margin=dict(l=70, r=60, t=70, b=60),
        title_text="Impulse Response Functions (IRF) from a VAR(3) process",
        title_x=0.5,
        height=500,
        hovermode="x",
        legend=dict(x=0.9, y=0.95),
        xaxis_range=[-0.5, 41],
    )
    _fig12
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ---
    """)
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md("""
    ## 5. How to produce a nice plot
    """)
    return


@app.cell(hide_code=True)
def _(USdata, go):
    # to produce a plot named pic, with Quarters in the x-axis, and CPI and UR in the y-axis, using the USdata dataframe we should start by doing:

    pic = go.Figure()
    pic.add_trace(
        go.Scatter(
            x=USdata.Quarters,
            y=USdata.CPI,
            mode="markers+lines",
            name="CPI",
            marker=dict(symbol="circle", size=5, color="blue"),
            line=dict(width=0.5),
        )
    )
    # Then we can add new features to our plot by using a function in Plotly: add_trace
    pic.add_trace(
        go.Scatter(
            x=USdata.Quarters,
            y=USdata.UR,
            mode="markers+lines",
            name="UR",
            marker=dict(symbol="circle", size=5, color="maroon"),
            line=dict(width=0.5),
        )
    )

    # Then we can add new features to our plot by using Plotly's function: update_layout
    pic.update_layout(
        margin=dict(l=70, r=60, t=70, b=60),
        title_text="My fancy title",
        title_x=0.5,
        height=500,
        width=980,
        hovermode="x unified",
        xaxis=dict(
            title="Quarterly observations",
            range=["1959-04-01", "2024-10-01"],
        ),
        yaxis=dict(title="Percentage points", range=[-2, 18]),
        font_family="Calibri",
        font_size=15,
    )
    pic
    return


@app.cell(hide_code=True)
def _(mo):
    mo.md(r"""
    ## 6. Auxiliary files (do not remove them)
    """)
    return


@app.cell(hide_code=True)
def _(pd):
    # Making data frames looking like in Julia and Pluto (first 5 rows + last row)
    def preview(df, n=5):
        if len(df) <= n + 1:
            return df

        ellipsis = pd.DataFrame(
            [["..."] * df.shape[1]],
            columns=df.columns,
            index=["…"],
        )

        return pd.concat(
            [
                df.head(n),
                ellipsis,
                df.tail(1),
            ]
        )

    return (preview,)


@app.cell(hide_code=True)
def _(mo):
    # Global CSS for kbd elements throughout the notebook
    mo.Html("""
        <style>
            kbd {
                background-color: #505050 !important;
                color: #fff !important;
                padding: 3px 6px;
                border-radius: 4px;
                font-family: monospace;
                font-size: 0.9em;
                border: 0px solid #666;
            }
        </style>
    """)
    return


@app.cell
def _():
    return


if __name__ == "__main__":
    app.run()
